/*
 * Judging program for "Street Directions"
 *
 * Author: Howard Cheng
 * Date: Nov 2, 1998
 *
 * Comment: This program checks the output of a program to see if it
 *          correct.  The following are checked:
 *
 *          1. out of range vertex numbers
 *          2. duplicate edges in output
 *          3. incorrect vertex degree
 *          4. solution contains extra streets not in the original map
 *          5. some streets have no direction assigned
 *          6. incorrect number of two-way streets
 *          7. reachability from any intersection to any other intersections
 *          8. extraneous output at the end (not spaces)
 *          9. format error
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>

#define FORMAT 2
#define REJECT 1
#define ACCEPT 0

#define MAX_N 1000
#define MAX_DEG 4
#define MAX_M MAX_DEG*MAX_N/2

int n, m;
int dfs[MAX_N+1];
int back[MAX_N+1];
int n_stack, dfn;
int v_stack[MAX_M];
int w_stack[MAX_M];
int two_way1, two_way2;
int case_no;
int format_err;

void report(int code)
{
  if (code == REJECT) {
    printf("Failed case %d\n", case_no);
  } else if (code == FORMAT) {
    printf("Format error\n");
  } else {
    assert(code == ACCEPT);
    printf("Accept\n");
  }
  exit(code);
}

int min(int x, int y)
{
  return (x < y) ? x : y;
}

int DFS(int deg[MAX_N+1], int adj[MAX_N+1][MAX_DEG], int src, int prev, 
	int visited[MAX_N+1])
{
  int i, w;

  if (visited[src]) {
    return 0;
  }
  if (1 <= src && src <= prev) {
    return 1;
  }
  visited[src] = 1;
  for (i = 0; i < deg[src]; i++) {
    w = adj[src][i];
    if (DFS(deg, adj, w, prev, visited)) {
      return 1;
    }
  }
  return 0;
}

int check_reachability(int deg[MAX_N+1], int adj[MAX_N+1][MAX_DEG])
{
  int visited[MAX_N+1];
  int i, j;
 
  for (i = 1; i <= n; i++) {
    visited[i] = 0;
  }
  DFS(deg, adj, 1, 0, visited);
  for (i = 1; i <= n; i++) {
    if (!visited[i]) {
      return 0;
    }
  }
  for (i = 2; i <= n; i++) {
    for (j = 1; j <= n; j++) {
      visited[j] = 0;
    }
    if (!DFS(deg, adj, i, i-1, visited)) {
      return 0;
    }
  }
  return 1;
}

int read_data(FILE *in, int deg[MAX_N+1], int adj[MAX_N+1][MAX_DEG])
{
  int i, u, v, t;
  
  t = fscanf(in, "%d %d", &n, &m);
  assert(t == 2);
  
  if (n == 0 && m == 0) {
    return 0;
  }
  assert(2 <= n && n <= MAX_N);
  assert(1 <= m && m <= MAX_M);    
  
  for (i = 1; i <= n; i++) {
    deg[i] = 0;
  }
  for (i = 0; i < m; i++) {
    t = fscanf(in, "%d %d", &u, &v);
    assert(t == 2);
    assert(1 <= u && u <= n);
    assert(1 <= v && v <= n);
    assert(u != v);
    for (t = 0; t < deg[u]; t++) {
      assert(adj[u][t] != v);
    }
    for (t = 0; t < deg[v]; t++) {
      assert(adj[v][t] != u);
    }
    adj[u][deg[u]++] = v;
    adj[v][deg[v]++] = u;
    assert(0 <= deg[u] && deg[u] <= 4);
    assert(0 <= deg[v] && deg[v] <= 4);
  }

  assert(check_reachability(deg, adj));
  return 1;
}

int eat_space(FILE *in, int *newline, int *end)
{
  int c, count;
  
  count = *newline = 0;
  while ((c = fgetc(in)) != EOF && isspace(c)) {
    if (c == '\n') {
      (*newline)++;
    }
    count++;
  }
  if (c != EOF) {
    ungetc(c, in);
    *end = 0;
  } else {
    *end = 1;
  }
  return count;
}

void read_soln(FILE *in, int deg[MAX_N+1], int adj[MAX_N+1][MAX_DEG],
	       int case_no)
{
  int i, u, v, t, space, newline, done, end, c;

  for (i = 1; i <= n; i++) {
    deg[i] = 0;
  }
  
  space = eat_space(in, &newline, &end);
  if (space > 0) {
    format_err = 1;
    printf("too much space before case number\n");
  }
  fscanf(in, "%d", &u);
  if (case_no != u) {
    /* wrong case number */
    printf("Incorrect case number\n");
    report(REJECT);
  }

  space = eat_space(in, &newline, &end);
  if (newline != 2 || space != 2) {
    format_err = 1;
    printf("wrong spacing after case number\n");
  }

  c = fgetc(in);
  if (c == '#') {
    done = 1;
    space = eat_space(in, &newline, &end);
    if (space != 1 || newline != 1) {
      printf("incorrect spacing after #\n");
      format_err = 1;
    }
  } else {
    ungetc(c, in);
    done = 0;
  }
  while (!done) {
    fscanf(in, "%d", &u);
    space = eat_space(in, &newline, &end);
    if (newline != 0 || space != 1) {
      format_err = 1;
      printf("too many spaces between vertex pair\n");
    }
    fscanf(in, "%d", &v);

    if (!(1 <= u && u <= n && 1 <= v && v <= n && u != v)) {
      /* out of range */
      printf("Vertices out of range\n");
      report(REJECT);
    }
    for (t = 0; t < deg[u]; t++) {
      if (adj[u][t] == v) {
	/* duplicate edges */
	printf("Duplicate edge\n");
	report(REJECT);
      }
    }
    adj[u][deg[u]++] = v;
    if (!(0 <= deg[u] && deg[u] <= 4)) {
      /* wrong degree */
      printf("Wrong degree\n");
      report(REJECT);
    }

    space = eat_space(in, &newline, &end);
    if (end) {
      done = 1;
      format_err = 1;
      printf("missing # after end of output\n");
    } else if (newline == 1) {
      if (space != 1) {
	format_err = 1;
	printf("too many spaces after a newline\n");
      }
      c = fgetc(in);
      if (c == '#') {
	done = 1;
	space = eat_space(in, &newline, &end);
	if (space != 1 || newline != 1) {
	  printf("incorrect spacing after #\n");
	  format_err = 1;
	}
      } else {
	if (c != EOF) {
	  ungetc(c, in);
	}
	done = 0;
      }
    } else if (newline < 1) {
      format_err = 1;
      printf("not end of line after an edge\n");
    } else {
      format_err = 1;
      printf("too many spaces/newlines within a case\n");
    }
  }
}

int check_edge(int deg[MAX_N+1], int adj[MAX_N+1][MAX_DEG], int u, int v)
{
  int i;
  for (i = 0; i < deg[u]; i++) {
    if (adj[u][i] == v) {
      return 1;
    }
  }
  return 0;
}

void check_edges(int deg1[MAX_N+1], int adj1[MAX_N+1][MAX_DEG],
		 int deg2[MAX_N+1], int adj2[MAX_N+1][MAX_DEG])
{
  int u, i;

  /* check that all edges in graph 2 are edges in graph 1 */
  for (u = 1; u <= n; u++) {
    for (i = 0; i < deg2[u]; i++) {
      if (!check_edge(deg1, adj1, u, adj2[u][i])) {
	/* extra edges in graph 2 */
	fprintf(stderr, "Solution graph contains extra edges\n");
	report(REJECT);
      }
    }
  }

  /* check that edges in graph 1 has been assigned a direction in graph 2 */
  for (u = 1; u <= n; u++) {
    for (i = 0; i < deg1[u]; i++) {
      if (!check_edge(deg2, adj2, u, adj1[u][i]) &&
	  !check_edge(deg2, adj2, adj1[u][i], u)) {
	/* unassigned edges */
	fprintf(stderr, "Unassigned edge\n");
	report(REJECT);
      }
    }
  }
}

void bicomp(int deg[MAX_N+1], int adj[MAX_N+1][MAX_DEG], int v, int pred)
{
  int i, w, count;

  dfn++;
  dfs[v] = back[v] = dfn;
  for (i = 0; i < deg[v]; i++) {
    w = adj[v][i];
    if (dfs[w] < dfs[v] && w != pred) {
      /* back edge or unexamined forward edge */
      v_stack[n_stack] = v;
      w_stack[n_stack] = w;
      n_stack++;
    }
    if (!dfs[w]) {
      bicomp(deg, adj, w, v);
      /* back up from recursion */
      if (back[w] >= dfs[v]) {
	/* new bicomponent */
	count = 0;
	while (v_stack[n_stack-1] != v || w_stack[n_stack-1] != w) {
	  assert(n_stack > 0);
	  n_stack--;
	  count++;
	}
	if (count == 0) {
	  /* we have a bicomponent containing only one edge */
	  two_way1++;
	}
	n_stack--;
      } else {
	back[v] = min(back[v], back[w]);
      }
    } else {
      /* w has been examined already */
      back[v] = min(back[v], dfs[w]);
    }
  }
}

int count_two_way(int deg[MAX_N+1], int adj[MAX_N+1][MAX_DEG])
{
  int res = 0;
  int u, i;

  for (u = 1; u <= n; u++) {
    for (i = 0; i < deg[u]; i++) {
      if (check_edge(deg, adj, adj[u][i], u)) {
	res++;
      }
    }
  }
  assert(res % 2 == 0);
  return res/2;
}

int main(int argc, char *argv[])
{
  int deg1[MAX_N+1];
  int adj1[MAX_N+1][MAX_DEG];
  int deg2[MAX_N+1];
  int adj2[MAX_N+1][MAX_DEG];
  FILE *data, *soln;
  int i, c;

  case_no = 1;
  format_err = 0;

  if (argc != 3) {
    fprintf(stderr, "Usage: %s <data> <soln>\n", argv[0]);
    exit(1);
  }
  if (!(data = fopen(argv[1], "r"))) {
    fprintf(stderr, "Cannot open data file.\n");
    exit(1);
  }
  if (!(soln = fopen(argv[2], "r"))) {
    fprintf(stderr, "Cannot open solution output file.\n");
    exit(1);
  }

  while (read_data(data, deg1, adj1)) {
    read_soln(soln, deg2, adj2, case_no);
    check_edges(deg1, adj1, deg2, adj2);

    /* compute number of two-way streets */
    two_way1 = n_stack = dfn = 0;
    for (i = 1; i <= n; i++) {
      dfs[i] = 0;
    }
    bicomp(deg1, adj1, 1, -1);
    two_way2 = count_two_way(deg2, adj2);
    if (two_way1 != two_way2) {
      /* incorrect number of two-way streets */
      printf("Incorrect number of two-way streets\n");
      report(REJECT);
    }
    
    if (!check_reachability(deg2, adj2)) {
      /* not reachable */
      printf("Not reachable\n");
      report(REJECT);
    }
    case_no++;
  }

  /* make sure that there is nothing extra at the end of the solution */
  c = fgetc(soln);
  if (c != EOF) {
    printf("Extraneous output at the end\n");
    report(REJECT);
  }

  
  fclose(data);
  fclose(soln);

  /* format error */
  if (format_err) {
    report(FORMAT);
  }

  /* everything works */
  report(ACCEPT);
return 0;
}
