#include <stdio.h>

/*
 * The points on the grid are read into the array points, and sorted
 * into increasing order (based first on increasing x values and
 * second on increasing y values).  Each point is then considered as
 * the starting point of a line (the sort order of the array ensures
 * that if the point is the first point then the output of lines will
 * be in the right order.
 *
 * For each point P considered, the array diffs is initialised to
 * contain for each subsequent point S the normalised delta x and
 * delta y pair from P to S.  This is done by dividing the actual x
 * and y deltas by the gcd of the two (or if either delta is 0 the
 * other is set to 1---they can't both be 0 as all points are
 * distinct).  If P is the start of a line then two normalised
 * delta pairs will exist with the same x delta and the same y
 * delta.  Diffs is sorted (same order as points) and checked for
 * adjacent deltas that are equal.  Details of a line are printed each time
 * equal deltas are found.
 */

#define IN_FILE "PROBLEMF.DAT"

#define MAXPOINTS 300
#define MAXLINE 256


struct point {
    int x, y;
};

struct diff {
    int dx, dy;
    int p2;
};

struct point points[MAXPOINTS];
struct diff diffs[MAXPOINTS];
int ind_diff[MAXPOINTS];

int num_points;
int num_diffs;

unsigned char bitmap[12000];

    static int
get_bit(int from, int to)
{
    long bit = (long) from * MAXPOINTS + to;
    int word = bit / 8;
    int offset = bit % 8;   
  
    return (bitmap[word] >> offset) & 1;
}  


    static void   
set_bit(int from, int to, int value)
{
    long bit = (long) from * MAXPOINTS + to;
    int word = bit / 8;
    int offset = bit % 8;

    if (value) {
        bitmap[word] |= 1 << offset;
    } else { 
        bitmap[word] &= ~(1 << offset);
    }
}

    static void
print_points(void)
{
    int i;
   
    for (i = 0; i < num_points; i++) {
	printf("%5d %5d\n", points[i].x, points[i].y);
    }
}


    static void
print_diffs(void)
{
    int i, j;
   
    for (i = 0; i < num_diffs; i++) {
	printf(" %03d(%4d,%4d)", diffs[i].p2, diffs[i].dx, diffs[i].dy);
    }
    printf("\n");
}


    static int
read_points(FILE *in_file)
{
    int x, y;

    num_points = 0;
    while (1) {
	if (fscanf(in_file, "%d %d", &x, &y) != 2) {
	    fprintf(stderr, "Error reading points\n");
	    exit(1);
	}
	if (x == 0 && y == 0) {
	    return num_points > 0 ? 1 : 0;
	}
	points[num_points].x = x;
        points[num_points].y = y;
	num_points++;
    }
}


/*
 * Returns an int greater, equal, or less than 0 dependding on whether
 * (x1, y1) is greater than, equal to, or less than (x2, y2). */
    static int
compare(int x1, int y1, int x2, int y2)
{
    if (x1 == x2) {
	return y1 - y2;
    }
    return x1 - x2;
}


    static void
sort_points(void)
{
    int i, j, min;
    struct point tmp;

    /*
     * minimum selection sort
     */
    for (i = 0; i < num_points - 1; i++) {
	min = i;
	for (j = i + 1; j < num_points; j++) {
	    if (compare(points[j].x, points[j].y, points[min].x,
			points[min].y) < 0) {
		min = j;
	    }
	}
	tmp = points[min];
	points[min] = points[i];
	points[i] = tmp;
    }
}


    static int
gcd(int x, int y)
{
    int res;

    if (x == 0) return y;
    if (y == 0) return x;
    while (1) {
	if (x == y) return x;

	if (x > y) {
	    x -= y;
	} else {
	    y -= x;
	}
    }
}


#define abs(x) ((x) > 0 ? (x) : -(x))

    static void
do_diffs(int start)
{
    int i, j, d;
    
    num_diffs = 0;
    for (i = start + 1; i < num_points; i++, num_diffs++) {
	d = gcd(abs(points[start].x - points[i].x),
		abs(points[start].y - points[i].y));
	diffs[num_diffs].dx = (points[start].x - points[i].x) / d;
	diffs[num_diffs].dy = (points[start].y - points[i].y) / d;
	diffs[num_diffs].p2 = i;
    }
}


    static void
print_point(int i)
{
    printf("(%4d,%4d)", points[i].x, points[i].y);
}


    static int
diff_compare(int i, int j)
{
    int res1 = compare(diffs[i].dx, diffs[i].dy, diffs[j].dx, diffs[j].dy);

    if (res1 != 0) {
	return res1;
    }
    return compare(points[diffs[i].p2].x, points[diffs[i].p2].y,
		   points[diffs[j].p2].x, points[diffs[j].p2].y);
}

    static void
sort_diffs(int lo, int hi)
{
    int i, j, p;
    struct diff tmp;

    if (lo >= hi) {
	return;
    }
    i = lo;
    j = hi;
    p = i;
    while (i <= j) {
	while (diff_compare(i, p) < 0) {
	    i++;
	}
	while (diff_compare(j, p) > 0) {
	    j--;
	}
	if (i <= j) {
	    tmp = diffs[i];
	    diffs[i] = diffs[j];
	    diffs[j] = tmp;
	    i++;
	    j--;
	}
    }
    sort_diffs(lo, j);
    sort_diffs(i, hi);
}


/*
 * By sorting the diffs we lose the required line ordering.
 * This needs fixing.
 */

    static int
find_lines(void)
{
    int i, j, k, l, jstart, pos;
    int found = 0, num_lines;

    for (i = 0; i < num_points - 2; i++) {
	do_diffs(i);
	sort_diffs(0, num_diffs - 1);
	for (j = 0, num_lines = 0; j < num_diffs - 1; ) {
	    if (diffs[j].dx == diffs[j + 1].dx &&
		diffs[j].dy == diffs[j + 1].dy && !get_bit(i, diffs[j].p2)) {
		if (!found) {
		    printf("The following lines were found:\n");
		    found = 1;
		}
		num_lines++;

		for (pos = 0; pos < num_lines - 1 &&
		     diffs[j].p2 > diffs[ind_diff[pos]].p2; pos++) {
		    ;
		}
		for (k = num_lines - 1; k > pos; k--) {
		    ind_diff[k] = ind_diff[k-1];
		}
		ind_diff[pos] = j;
/*		print_point(i);
		print_point(diffs[j].p2);
		print_point(diffs[j+1].p2); */
		set_bit(diffs[j].p2, diffs[j+1].p2, 1);
		set_bit(diffs[j+1].p2, diffs[j].p2, 1);
		jstart = j;
		for (j += 2; j < num_diffs && diffs[j].dx == diffs[j-1].dx &&
		     diffs[j].dy == diffs[j-1].dy; j++) {
/*		    print_point(diffs[j].p2); */
		    for (l = jstart; l < j; l++) {
			set_bit(diffs[l].p2, diffs[j].p2, 1);
			set_bit(diffs[j].p2, diffs[l].p2, 1);
		    }
		}
/*		printf("\n");*/
	    } else {
		j++;
	    }
	}
	for (j = 0; j < num_lines; j++) {
	    print_point(i);
	    k = ind_diff[j];
	    print_point(diffs[k].p2);
	    for (k++; k < num_diffs && diffs[k].dx == diffs[k-1].dx &&
		 diffs[k].dy == diffs[k-1].dy; k++) {
		print_point(diffs[k].p2);
	    }
	    printf("\n");
	}
    }
    return found;
}


main()
{
    FILE *in_file;
    int i;

/*    if ((in_file = fopen(IN_FILE, "r")) == 0) {
	fprintf(stderr, "Couldn't open file %s\n", IN_FILE);
	exit(1);
    }*/
    in_file = stdin;

    while (read_points(in_file)) {
	sort_points();
/*print_points();*/
/*print_diffs();*/
	for (i = 0; i < sizeof(bitmap); i++) {
	    bitmap[i] = 0;
	}
	if (find_lines() == 0) {
	    printf("No lines were found\n");
	}
    }
    return 0;
}
