// #define _DEBUG
#include <assert.h>
#include <fstream.h>
#include <iostream.h>

// typedef unsigned int bool;
#define TRUE 1
#define FALSE 0

enum Direction {
	dirA = 0,
	dirB = 1,
	dirC = 2
};

inline int Max(int a, int b)
{ return a > b ? a : b; }


// --- Piece --------------------------------------------

struct Piece {
	int		fValue[3];	// value of piece for each direction
	bool	fTaken;		// flag whether pieces has been taken
};

static const Piece cEmptyPiece = { 0, 0, 0 };


// --- Row --------------------------------------------
// this struct is used for accessing the board row by row

struct Row {
	int fLen;			// length of the row
	int fIndices[5];	// index of the piece on the board
};

/*

The board and the indices for the pieces:

                 0
              1     2
           3     4     5
              6     7
           8     9    10
             11    12
          13    14    15
             16    17
                18
*/


//   cRows[ dir ][ n ] returns a row given the direction
//   dir and the number n ( 0 <= n < 5 ) of the row

static const Row cRows[3][5] = {
	{
		{ 3,  3,  8, 13, -1, -1 },
		{ 4,  1,  6, 11, 16, -1 },
		{ 5,  0,  4,  9, 14, 18 },
		{ 4,  2,  7, 12, 17, -1 },
		{ 3,  5, 10, 15, -1, -1 }
	},
	{
		{ 3,  0,  1,  3, -1, -1 },
		{ 4,  2,  4,  6,  8, -1 },
		{ 5,  5,  7,  9, 11, 13 },
		{ 4, 10, 12, 14, 16, -1 },
		{ 3, 15, 17, 18, -1, -1 }
	},
	{
		{ 3,  0,  2,  5, -1, -1 },
		{ 4,  1,  4,  7, 10, -1 },
		{ 5,  3,  6,  9, 12, 15 },
		{ 4,  8, 11, 14, 17, -1 },
		{ 3, 13, 16, 18, -1, -1 }
	}
};


// --- Game --------------------------------------------

class Game {
public:
	Game();
	const Piece& GetPieceAt(int index) const;	// Get piece on board
	void   PutPieceAt(int index, const Piece& piece);
										// Set piece onto board
	int    GetMaximum() const;			// Calculate possible maximum
	void   SetNumbers(int a0, int a1, int a2,
                      int b0, int b1, int b2,
                      int c0, int c1, int c2);
	                  
	const Piece& PieceFromSet(int index) const;		// Get piece from set of pieces
	bool   Taken(int index);
	void   SetTaken(int index, bool value);
	
#ifdef _DEBUG
	void  Print() const;
#endif

protected:
	int   GetRowMaximum(Direction dir, int rowNumber) const;
	
protected:
	Piece	fBoard[19];		// the 19 fields on the board
	Piece	fPieceSet[27];	// the 27 unique pieces
	int		fNumbers[3][3];	// the 3 numbers for the 3 directions
	int		fDirMax[3];		// maximum number for each direction
};

Game::Game()
{
	for (int i = 0; i < 19; i++)
		fBoard[i] = cEmptyPiece;
}

inline const Piece& Game::GetPieceAt(int index) const
{
	assert( index >= 0 && index < 19 );
	return fBoard[index];
}

inline void Game::PutPieceAt(int index, const Piece& piece)
{
	assert( index >= 0 && index < 19 );
	fBoard[index] = piece;
}

inline const Piece& Game::PieceFromSet(int index) const
{
	assert( index >= 0 && index < 27 );
	return fPieceSet[index];
}

inline bool Game::Taken(int index)
{
	assert( index >= 0 && index < 27 );
	return fPieceSet[index].fTaken;
}

inline void Game::SetTaken(int index, bool value)
{
	assert( index >= 0 && index < 27 );
	fPieceSet[index].fTaken= value;
}

int Game::GetMaximum() const
{
	// Note: For a full board, this method returns the actual score
	
	int score= 0;
	for (int i= 0; i < 5; i++)
		for (int dir= dirA; dir <= dirC; dir++) {
			score += GetRowMaximum((Direction)dir, i);
			if (score < 0)
				return -1;
		}
	return score;
}

int Game::GetRowMaximum(Direction dir, int rowNumber) const
{
	// The possible maximum for a row is calculated as follows:
	// - if one or more pieces have been set in this row and
	//   they have the same number, the maximum is number * row_length
	// - if no piece has been set, we take the maximum number for this
	//   direction; we assume the values in cValueSet are sorted and
	//   take the first element (maximum = row_max_number * row_length)
	// - if two or more pieces have been set in this row and
	//   they have different numbers, we return a huge negative number
	
	// Note: For a full row, this methods returns the actual score
	
	const Row& row = cRows[dir][rowNumber];	// current row
	int pieceValue = fBoard[ row.fIndices[0] ].fValue[dir];
			// number of first piece in row
	
	for (int i = 1; i < row.fLen; i++) {
		int v = fBoard[ row.fIndices[i] ].fValue[dir];
		if (pieceValue == 0)
			pieceValue = v;
		else if (v != pieceValue && v != 0)	// there are different numbers in this row
			return -19 * 3 * 8;
	}
	if (pieceValue == 0)
		pieceValue = fDirMax[dir];	// maximum number score for this direction
	return pieceValue * row.fLen;
}

void Game::SetNumbers(int a0, int a1, int a2,
                      int b0, int b1, int b2,
                      int c0, int c1, int c2)
{
	fNumbers[dirA][0] = a0;
	fNumbers[dirA][1] = a1;
	fNumbers[dirA][2] = a2;
	fNumbers[dirB][0] = b0;
	fNumbers[dirB][1] = b1;
	fNumbers[dirB][2] = b2;
	fNumbers[dirC][0] = c0;
	fNumbers[dirC][1] = c1;
	fNumbers[dirC][2] = c2;
	
	// take maximum
	fDirMax[dirA] = Max( a0, Max(a1, a2) );
	fDirMax[dirB] = Max( b0, Max(b1, b2) );
	fDirMax[dirC] = Max( c0, Max(c1, c2) );
	
	// generate the 27 pieces
	for (int i = 0; i < 3; i++)
		for (int j = 0; j < 3; j++)
			for (int k = 0; k < 3; k++) {
				fPieceSet[ (i * 3 + j) * 3 + k ].fValue[dirA] = fNumbers[dirA][i];
				fPieceSet[ (i * 3 + j) * 3 + k ].fValue[dirB] = fNumbers[dirB][j];
				fPieceSet[ (i * 3 + j) * 3 + k ].fValue[dirC] = fNumbers[dirC][k];
				fPieceSet[ (i * 3 + j) * 3 + k ].fTaken= FALSE;
			}
}


#ifdef _DEBUG

#include <stdio.h>

void Game::Print() const
{
	printf("\t\t\t\t%d_%d_%d\n\n",
			fBoard[0].fValue[dirA], fBoard[0].fValue[dirB], fBoard[0].fValue[dirC]);
	printf("\t\t%d_%d_%d\t\t\t%d_%d_%d\n\n",
			fBoard[1].fValue[dirA], fBoard[1].fValue[dirB], fBoard[1].fValue[dirC],
			fBoard[2].fValue[dirA], fBoard[2].fValue[dirB], fBoard[2].fValue[dirC]);
	printf("%d_%d_%d\t\t\t%d_%d_%d\t\t\t%d_%d_%d\n\n",
			fBoard[3].fValue[dirA], fBoard[3].fValue[dirB], fBoard[3].fValue[dirC],
			fBoard[4].fValue[dirA], fBoard[4].fValue[dirB], fBoard[4].fValue[dirC],
			fBoard[5].fValue[dirA], fBoard[5].fValue[dirB], fBoard[5].fValue[dirC]);
	printf("\t\t%d_%d_%d\t\t\t%d_%d_%d\n\n",
			fBoard[6].fValue[dirA], fBoard[6].fValue[dirB], fBoard[6].fValue[dirC],
			fBoard[7].fValue[dirA], fBoard[7].fValue[dirB], fBoard[7].fValue[dirC]);
	printf("%d_%d_%d\t\t\t%d_%d_%d\t\t\t%d_%d_%d\n\n",
			fBoard[8].fValue[dirA], fBoard[8].fValue[dirB], fBoard[8].fValue[dirC],
			fBoard[9].fValue[dirA], fBoard[9].fValue[dirB], fBoard[9].fValue[dirC],
			fBoard[10].fValue[dirA], fBoard[10].fValue[dirB], fBoard[10].fValue[dirC]);
	printf("\t\t%d_%d_%d\t\t\t%d_%d_%d\n\n",
			fBoard[11].fValue[dirA], fBoard[11].fValue[dirB], fBoard[11].fValue[dirC],
			fBoard[12].fValue[dirA], fBoard[12].fValue[dirB], fBoard[12].fValue[dirC]);
	printf("%d_%d_%d\t\t\t%d_%d_%d\t\t\t%d_%d_%d\n\n",
			fBoard[13].fValue[dirA], fBoard[13].fValue[dirB], fBoard[13].fValue[dirC],
			fBoard[14].fValue[dirA], fBoard[14].fValue[dirB], fBoard[14].fValue[dirC],
			fBoard[15].fValue[dirA], fBoard[15].fValue[dirB], fBoard[15].fValue[dirC]);
	printf("\t\t%d_%d_%d\t\t\t%d_%d_%d\n\n",
			fBoard[16].fValue[dirA], fBoard[16].fValue[dirB], fBoard[16].fValue[dirC],
			fBoard[17].fValue[dirA], fBoard[17].fValue[dirB], fBoard[17].fValue[dirC]);
	printf("\t\t\t\t%d_%d_%d\n\n",
			fBoard[18].fValue[dirA], fBoard[18].fValue[dirB], fBoard[18].fValue[dirC]);
};
#endif


// --- Backtracking --------------------------------------------

void Try(int index, int& max, Game& g);

void Try(int index, int& max, Game& g)
{
	int m;
	
	for (int i = 0; i < 27; i++)	// try all 27 pieces
	
		// if piece has not yet been assigned to a field ...
		if (! g.Taken(i)) {
		
			// put the piece onto the board
			g.PutPieceAt(index, g.PieceFromSet(i));
			g.SetTaken(i, TRUE);
			
			// if the possible max is greater than the highes score so far
			m = g.GetMaximum();
			if (m > max) {

				// board isn't full yet
				if (index < 18) {
					Try(index + 1, max, g);	// recursively try next piece
				
				// the board is full and we have a new maximum
				} else {
					max = m;
#ifdef _DEBUG
					printf("New max: %d\n\n", max);
					g.Print();
#endif
				}
			}
			
			// restore board (backtracking)
			g.SetTaken(i, FALSE);
			g.PutPieceAt(index, cEmptyPiece);
		}
}
			

// --- Closed Formula -------------------------------------------

int  ClosedFormula(int a0, int a1, int a2,
                   int b0, int b1, int b2,
                   int c0, int c1, int c2);

inline void Swap(int& a, int& b)
{ int tmp; tmp= a; a= b; b= tmp; }

int  ClosedFormula(int a0, int a1, int a2,
                   int b0, int b1, int b2,
                   int c0, int c1, int c2)
{
	// sort
	if (a0 < a1) Swap(a0, a1);
	if (a0 < a2) Swap(a0, a2);
	if (a1 < a2) Swap(a1, a2);
	if (b0 < b1) Swap(b0, b1);
	if (b0 < b2) Swap(b0, b2);
	if (b1 < b2) Swap(b1, b2);
	if (c0 < c1) Swap(c0, c1);
	if (c0 < c2) Swap(c0, c2);
	if (c1 < c2) Swap(c1, c2);
	
	int v1, v2, v3;
	v1 =  8 * a0 + 6 * a1 + 5 * a2
		+ 7 * b0 + 7 * b1 + 5 * b2
		+ 7 * c0 + 7 * c1 + 5 * c2;
		
	v2 =  8 * b0 + 6 * b1 + 5 * b2
		+ 7 * a0 + 7 * a1 + 5 * a2
		+ 7 * c0 + 7 * c1 + 5 * c2;
		
	v3 =  8 * c0 + 6 * c1 + 5 * c2
		+ 7 * a0 + 7 * a1 + 5 * a2
		+ 7 * b0 + 7 * b1 + 5 * b2;
	
	return Max(v1, Max(v2, v3) );
}

// --- main -----------------------------------------------------

int main(int, char*[])
{
	ifstream is("hexagon.in");
//	ofstream os("hexagon.out");
	
	int currProblem, nProblems;
	for (currProblem = 1, is >> nProblems; nProblems > 0; 
	     currProblem++, nProblems--) {
		Game g;
		int a0, a1, a2, b0, b1, b2, c0, c1, c2;
		is >> a0 >> a1 >> a2 >> b0 >> b1 >> b2 >> c0 >> c1 >> c2;
		g.SetNumbers(a0, a1, a2, b0, b1, b2, c0, c1, c2);
		
		int max= 0;
		//Try(0, max, g);
		//assert(max == ClosedFormula(a0, a1, a2, b0, b1, b2, c0, c1, c2));
		cout << "Test #" << currProblem << endl;
		cout << ClosedFormula(a0, a1, a2, b0, b1, b2, c0, c1, c2) << endl << endl;
		//cout << max << endl << endl;
	}
	
	return 0;
}

