/* $Id: ships.c,v 1.4 1996/11/01 15:23:59 strebel Exp $ */

/*--------------------------------------------------------------

program to solve the following problem:

There's a grid containing a pattern of 'x', composed of
the following basic patterns, which may be arbitrarily
rotated and translated (but not reflected):

  xx  xx    xx  x      x   x
  xx,  xx, xx , xxx, xxx, xxx, xxxx

The input consists of incomplete information on the grid,
namely the symbols 'x','o' and '.' represent

  'x' there is a 'x'
  'o' there is no 'x'
  '.' dunno

You have to find out if the grid can always be uniquely
determined with at most 1 miss, i.e., you may uncover
the '.' as long as you get at most one 'o'.

Your output is either

  'no solution.' if there's no such grid possible
  'yes.'         if the grid is unique with at most one miss
  'no.'          otherwise

Example Input.

10 10
.x..x.....
oooooxoooo
oxooxxx...
xxoooooo..
xoooxooo..
ooxxxxoo..
oooooxxoox
ooooooxoox
ooooooooxx
oooooooooo

0 0

Example Output.

yes.

Limitations.

The grid size is <= 16.

--------------------------------------------------------------*/

#define D_TRYS (1<<0)
#define D_INPUT (1<<1)
#define D_SHIPS (1<<2)
#define D_SOLS (1<<3)

#define DEBUG (0)

#include <assert.h>
#include <stdio.h>
#include <strings.h>

#define MAXSIZE 16 /* maximum size of grid */
#define NOFSHIP 7  /* number of ships */
#define AREASHIP NOFSHIP*4 /* area covered by ships */
#define MAXMATS ((AREASHIP) + 2)
  /* this deserves some explanation: if one has >= 30 possible
     solutions, it is not possible to get along with one miss.
     This limit is essential if there are lots of solutions. 
     
     The reason is the following: we may throw away a possible
     solution (call it s_1) if and only if we find a point p_1, 
     where s_1 has no ship, but all other s_k have a ship there.
     We can then continue with (s_2,p_2), ...
     Now, obviously, p_i != p_k for i != k, and we know that
     solution s_k has ships on points p_1,p_2,...p_{k-1}. Now,
     since a solution has AREASHIP points covered by ships, we
     may exclude at most AREASHIP solutions by this method.

     It follows that there's no unique solution for more than
     AREASHIP+1 solutions. Phew, q.e.d. */

#define MAX(a,b) ((a)>=(b)?(a):(b))

typedef struct {
  int row, col;
  int f[MAXSIZE];
} Field;

/* basic operations */

Field empty (void)
{
  Field c;
  int i;

  for (i = 0; i < MAXSIZE; i++) c.f[i] = 0;
  c.row = c.col = 0;
  return (c);
}

Field dot (void)
{
  Field c;

  c = empty();
  c.f[0] = 1;
  c.row = c.col = 1;
  return (c);
}

int isempty (Field a)
{
  int i;
  int val = 0;
  
  for (i = 0; i < MAXSIZE; i++) val |= a.f[i];
  return (val == 0);
}

Field not (Field a)
{
  Field c;
  int i;

  c.row = a.row;
  c.col = a.col;
  for (i = 0; i < MAXSIZE; i++) c.f[i] = ~a.f[i];
  return (c);
}

Field and (Field a, Field b)
{
  Field c;
  int i;

  c.row = MAX(a.row,b.row);
  c.col = MAX(a.col,b.col);
  for (i = 0; i < MAXSIZE; i++) c.f[i] = a.f[i]&b.f[i];
  return (c);
}

Field or (Field a, Field b)
{
  Field c;
  int i;

  c.row = MAX(a.row,b.row);
  c.col = MAX(a.col,b.col);
  for (i = 0; i < MAXSIZE; i++) c.f[i] = a.f[i]|b.f[i];
  return (c);
}

Field eor (Field a, Field b)
{
  Field c;
  int i;

  c.row = MAX(a.row,b.row);
  c.col = MAX(a.col,b.col);
  for (i = 0; i < MAXSIZE; i++) c.f[i] = a.f[i]^b.f[i];
  return (c);
}

int eq (Field a, Field b)
{
  return isempty (eor(a,b));
}

Field right (Field a)
{
  Field c;
  int i;

  c.row = a.row;
  c.col = a.col + 1;
  for (i = 0; i < MAXSIZE; i++) c.f[i] = a.f[i]<<1;
  return (c);
}

Field down (Field a)
{
  Field c;
  int i;

  c.row = a.row + 1;
  c.col = a.col;
  c.f[0] = 0;
  for (i = 1; i < MAXSIZE; i++) c.f[i] = a.f[i-1];
  return (c);
}

/* input output */

void get (Field *hit, Field *nohit)
{
  char buf[MAXSIZE+1];
  int i;
  int row, col;
  int pat;
  char *p;

  *hit = *nohit = empty();
  scanf ("%i %i\n", &row, &col);
  for (i = 0; i < row; i++) {
    scanf ("%s\n", buf);
    for (p = buf,pat = 1; *p; p++,pat<<=1) {
      if (*p == 'x') (*hit).f[i] |= pat;
      else if (*p == 'o') (*nohit).f[i] |= pat;
      /* fi */
    }
  }
  (*hit).row = (*nohit).row = row;
  (*hit).col = (*nohit).col = col;
}

void put (Field hit, Field nohit)
{
  char buf[MAXSIZE+1];
  int row = hit.row, col = hit.col;
  int i,j;
  int pat;
  char *p;

  for (i = 0; i < row; i++) {
    for (j = 0,pat = 1; j < col; j++,pat<<=1) {
      if (pat & hit.f[i]) printf ("x");
      else if (pat & nohit.f[i]) printf ("o");
      else printf (".");
      /* fi */
    }
    printf ("\n");
  }
  printf ("\n");
}

/* main code */

/* below are these ships, with rotations clockwise
x.. ..x xx xx. .xx .x. xxxx
xxx xxx	xx .xx xx. xxx
*/

static struct {
  int nrot;
  Field rot[4];
} ship[NOFSHIP] = {
  {4,{{2,3,{0x01,0x07}},
      {3,2,{0x03,0x01,0x01}},
      {2,3,{0x07,0x04}},
      {3,2,{0x02,0x02,0x03}}}},
  {4,{{2,3,{0x04,0x07}},
      {3,2,{0x01,0x01,0x03}},
      {2,3,{0x07,0x01}},
      {3,2,{0x03,0x02,0x02}}}},
  {1,{{2,2,{0x03,0x03}}}},
  {2,{{2,3,{0x03,0x06}},
      {3,2,{0x02,0x03,0x01}}}},
  {2,{{2,3,{0x06,0x03}},
      {3,2,{0x01,0x03,0x02}}}},
  {4,{{2,3,{0x02,0x07}},
      {3,2,{0x01,0x03,0x01}},
      {2,3,{0x07,0x02}},
      {3,2,{0x02,0x03,0x02}}}},
  {2,{{1,4,{0x0f}},
      {4,1,{0x01,0x01,0x01,0x01}}}}
};

static Field hit, nohit;

/* list of successful tries */

int nofmats;
static Field mat[MAXMATS];

void init (void)
{
  /* nothing to do for ships */
  nofmats = 0;
}

int full (void)
{
  return (nofmats == MAXMATS);
}

/* include without duplicates */
void include (Field a)
{
  int i;

  if (full()) return;
  for (i = 0; i < nofmats; i++) {
    if (eq (a, mat[i])) return;
  }
  mat[nofmats++] = a;
}

void exclude (int i)
{
  mat[i] = mat[--nofmats];
}

void try (int s, Field have, Field avoid)
{
  Field tmp, tmp1, tmp2;
  int r;
  int i, j;

#if DEBUG & D_TRYS
  {
    printf ("%i\n",s);
    put (have, avoid);
  }
#endif
  if (s >= NOFSHIP) {
    if (isempty(and(not(have),hit))) {
      include (have);
    }
  } else {
    for (r = 0; r < ship[s].nrot; r++) {
      tmp = ship[s].rot[r];
      tmp1 = tmp;
      for (i = 0; i <= hit.row - tmp.row; i++) {
	tmp2 = tmp1;
	for (j = 0; j <= hit.col - tmp.col; j++) {
	  if (full()) return;
	  if (isempty(and(avoid,tmp2))) {
	    try (s+1, or(have,tmp2), or(avoid,tmp2));
	  }
	  tmp2 = right(tmp2);
	}
	tmp1 = down(tmp1);
      }
    }
  }
}

void cancel (void)
{
  Field tmp, tmp1, tmp2;
  int i, j;
  int change = 1;
  int idx, n;

  while (change) {
    change = 0;
    tmp = dot();
    tmp1 = tmp;
    for (i = 0; i <= hit.row - tmp.row; i++) {
      tmp2 = tmp1;
      for (j = 0; j <= hit.col - tmp.col; j++) {
	for (n = 0, idx = -1; n < nofmats; n++) {
	  if (isempty(and(tmp2,mat[n]))) {
	    if (idx >= 0) { idx = -1; break; }
	    idx = n;
	  }
	}
	if (idx >= 0) {
	  exclude (idx);
	  change = 1;
	}
	tmp2 = right(tmp2);
      }
      tmp1 = down(tmp1);
    }
  }
}

void main (int argc, char **argv)
{
#if DEBUG & D_SHIPS
  {
    int i, j;

    for (i = 0; i < NOFSHIP; i++) {
      for (j = 0; j < ship[i].nrot; j++) {
	put (ship[i].rot[j], empty());
      }
    }
  }
#endif
  while (1) {
    get (&hit, &nohit);
    if (hit.row == 0) break;
#if DEBUG & D_INPUT
    {
      put (hit, nohit);
    }
#endif
    init ();
    try (0, empty(), nohit);
#if DEBUG & D_SOLS
    {
      int i;

      printf ("--------------------------------\n");
      for (i = 0; i < nofmats; i++) {
	put (mat[i], empty());
      }
    }
#endif
    if (nofmats == 0) {
      printf ("no solution.\n");
    } else {
      cancel();
      printf ((nofmats == 0)?("yes.\n"):("no.\n"));
    }
  }
}
